﻿using CompositeMementoSample.Infrastructure;
using CompositeMementoSample.Models;
using Microsoft.Practices.Prism.Commands;
using Microsoft.Practices.Prism.Mvvm;
using Microsoft.Practices.Prism.ViewModel;
using System.Collections.Generic;
using System.Collections.ObjectModel;
using System.Windows.Input;

namespace CompositeMementoSample.ViewModels
{
    public class FileGroupViewModel : BindableBase
    {
        ViewMode viewMode;
        public ViewMode ViewMode
        {
            get { return viewMode; }
            set { SetProperty(ref viewMode, value); }
        }

        FileGroupModel activeFileGroup;
        public FileGroupModel ActiveFileGroup
        {
            get
            {
                return this.activeFileGroup;
            }

            set
            {
                SetProperty(ref activeFileGroup, value);
                OnPropertyChanged(() => CompositeList);
                if (saveXmlCommand != null)
                    saveXmlCommand.RaiseCanExecuteChanged();
            }
        }

        public ReadOnlyCollection<object> CompositeList 
        { 
            get 
            { 
                return ActiveFileGroup == null ? new ReadOnlyCollection<object>(new List<object>()) : ActiveFileGroup.CompositeList; 
            } 
        }

        IOpenDialog OpenDialogService;
        ISaveDialog SaveDialogService;

        public FileGroupViewModel(IOpenDialog OpenDialogService, ISaveDialog SaveDialogService)
        {
            this.ActiveFileGroup = new FileGroupModel();
            this.OpenDialogService = OpenDialogService;
            this.SaveDialogService = SaveDialogService;

            OpenDialogService.Filter = ActiveFileGroup.OpenFileDialogFilter;
            SaveDialogService.Filter = ActiveFileGroup.OpenFileDialogFilter;

            saveXmlCommand = new DelegateCommand(SaveXml, CanSaveXml);
            loadXmlCommand = new DelegateCommand(LoadXml);
            addFilesCommand = new DelegateCommand(AddFiles);
        }

        DelegateCommand saveXmlCommand;
        public ICommand SaveXmlCommand
        {
            get { return saveXmlCommand; }
        }

        bool CanSaveXml()
        {
            return ActiveFileGroup != null && ActiveFileGroup.CompositeList.Count > 0;
        }

        void SaveXml()
        {
            if (SaveDialogService.ShowSaveFileDialog() != true)
                return;

            ActiveFileGroup.WriteToFile(SaveDialogService.SaveFileName);
        }

        DelegateCommand loadXmlCommand;
        public ICommand LoadXmlCommand
        {
            get { return loadXmlCommand; }
        }

        void LoadXml()
        {
            OpenDialogService.Multiselect = false;
            if (OpenDialogService.ShowOpenFileDialog() != true)
                return;

            FileGroupModel newModel = FileGroupModel.ReadFromFile(OpenDialogService.OpenFileName);
            if (newModel == null)
                return;

            ActiveFileGroup = newModel;
        }

        DelegateCommand addFilesCommand;
        public ICommand AddFilesCommand
        {
            get { return addFilesCommand; }
        }

        void AddFiles()
        {
            OpenDialogService.Multiselect = true;
            if (OpenDialogService.ShowOpenFileDialog() != true)
                return;

            foreach (string path in OpenDialogService.OpenFileNames)
            {
                if (path.EndsWith(ActiveFileGroup.SerializedExtension))
                {
                    try
                    {
                        FileGroupModel fgm = FileGroupModel.ReadFromFile(path);
                        if (fgm != null)
                        {
                            if (string.IsNullOrEmpty(fgm.Name))
                            {
                                fgm.Name = System.IO.Path.GetFileNameWithoutExtension(path) + " (" + path + ")";
                            }
                            ActiveFileGroup.Groups.Add(fgm);
                            continue;
                        }
                    }
                    //if we get an exception assume it's not a FileGroupModel and add as a regular file
                    catch { }
                }
                ActiveFileGroup.Files.Add(new System.IO.FileInfo(path));
            }
            OnPropertyChanged(() => CompositeList);
            saveXmlCommand.RaiseCanExecuteChanged();
        }
    }
}
